<?php
require __DIR__.'/vendor/autoload.php';
use Google\Client;
use Google\Service\Indexing;

/* ---------- إعداد قاعدة SQLite لمنع التكرار ---------- */
function db(): PDO {
    static $pdo;
    if (!$pdo) {
        $pdo = new PDO('sqlite:' . __DIR__.'/sent_urls.db');
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->exec("CREATE TABLE IF NOT EXISTS sent(
                       url TEXT PRIMARY KEY,
                       last_type TEXT NOT NULL,
                       sent_at INTEGER NOT NULL)");
    }
    return $pdo;
}

function alreadySent(string $url, string $type): bool {
    $q = db()->prepare('SELECT 1 FROM sent WHERE url=? AND last_type=?');
    $q->execute([$url, $type]);
    return (bool)$q->fetchColumn();
}

function markSent(string $url, string $type): void {
    $q = db()->prepare('INSERT OR REPLACE INTO sent VALUES(?,?,?)');
    $q->execute([$url, $type, time()]);
}

/* ---------- تهيئة عميل Google مرّة واحدة ---------- */
function googleService(): Indexing {
    static $svc;
    if (!$svc) {
        $c = new Client();
        $c->setAuthConfig('/home/abo2/public_html/indexurls/indexing.json'); // عدّل المسار حسب الحاجة
        $c->addScope('https://www.googleapis.com/auth/indexing');
        $svc = new Indexing($c);
    }
    return $svc;
}

/* ---------- إرسال رابط ---------- */
function sendUrl(string $url, string $type = 'URL_UPDATED'): bool {
    $svc = googleService();
    $req = new Indexing\UrlNotification(['url' => $url, 'type' => $type]);
    $svc->urlNotifications->publish($req);
    markSent($url, $type);
    return true;
}

/* ---------- قراءة خلاصة وإرجاع الروابط ---------- */
function extractFeedLinks(string $feedUrl): array {
    $links = [];
    $ch = curl_init($feedUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => 1,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_USERAGENT => 'IndexingCron/1.0'
    ]);
    $xmlStr = curl_exec($ch);
    curl_close($ch);
    if (!$xmlStr) return [];

    libxml_use_internal_errors(true);
    $xml = simplexml_load_string($xmlStr);
    if (!$xml) return [];

    foreach ($xml->channel->item as $it) {
        $links[] = (string)$it->link;
    }

    foreach ($xml->entry as $entry) {
        foreach ($entry->link as $lnk) {
            $a = $lnk->attributes();
            if (!isset($a['rel']) || $a['rel'] == 'alternate') {
                $links[] = (string)$a['href'];
                break;
            }
        }
    }

    return array_unique(array_filter(array_map('trim', $links), 'strlen'));
}

/* ---------- التحقق من صحة الرابط بما يدعم الروابط العربية ---------- */
function isValidUrl(string $url): bool {
    $encoded = preg_replace_callback(
        '/[^\x00-\x7F]+/',
        function ($m) {
            return urlencode($m[0]);
        },
        $url
    );
    return filter_var($encoded, FILTER_VALIDATE_URL) !== false;
}


function dailySentCount(): int {
    $startOfDay = strtotime('today midnight');
    $q = db()->prepare('SELECT COUNT(*) FROM sent WHERE sent_at >= ?');
    $q->execute([$startOfDay]);
    return (int)$q->fetchColumn();
}

